package ga.view.aerial;

import ga.core.individual.ICostInfo;
import ga.core.individual.IIndividual;
import ga.view.appstate.menu.IMenuListenerParent;
import ga.view.appstate.menu.MenuListener;

import java.util.logging.Level;
import java.util.logging.Logger;

import com.jme3.math.FastMath;

import de.lessvoid.nifty.screen.DefaultScreenController;

/**
 * This is a controller that allows communication with the nifty gui.
 * 
 * @param <T>
 *          The generic type of the individuals.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class AerialUIController<T extends IIndividual<T>> extends
    DefaultScreenController implements IMenuListenerParent {

  // the logger for this class
  /** The Constant LOGGER. */
  private static final Logger LOGGER = Logger
      .getLogger(AerialUIController.class.getName());

  /** The evaluator. */
  private final AerialEvaluationState<T> evaluator;

  /** The individual. */
  private T individual;

  /** The menu listener. */
  private MenuListener menuListener;

  /**
   * Instantiates a new aerial ui controller.
   * 
   * @param evaluator
   *          the evaluator
   */
  public AerialUIController(final AerialEvaluationState<T> evaluator) {
    this.evaluator = evaluator;
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * ga.view.appstate.menu.IMenuListenerParent#setMenuListener(ga.view.appstate
   * .menu.MenuListener)
   * 
   * @since 12.08.2012
   * 
   * @author Stephan Dreyer
   */
  @Override
  public void setMenuListener(final MenuListener menuListener) {
    this.menuListener = menuListener;
  }

  /**
   * Sets the individual.
   * 
   * @param individual
   *          the new individual
   */
  public void setIndividual(final T individual) {
    this.individual = individual;
  }

  /**
   * Callback method. Called from the gui.
   * 
   * @param s
   *          The string.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void onFitnessChange(final String s) {
    final double newFitness = Double.valueOf(s);

    if (LOGGER.isLoggable(Level.FINE)) {
      LOGGER.fine("Fitness: " + newFitness);
    }

    if (individual != null) {
      individual.setFitness(newFitness);

      evaluator.fireIndividualEvaluated(individual);

      evaluator.fireNewIndividualRequested();
    }
  }

  /**
   * Callback method. Called from the gui.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void onTerminate() {
    onEndScreen();

    if (menuListener != null) {
      menuListener.nextState();
    }
  }

  /**
   * Gets the id.
   * 
   * @return the id
   */
  public String getId() {
    return individual != null ? String.valueOf(individual.getId()) : "";
  }

  /**
   * Gets the costs.
   * 
   * @return the costs
   */
  public String getCosts() {
    if (individual != null) {
      if (individual instanceof ICostInfo) {
        return ((ICostInfo) individual).getCostString();
      }
    }
    return "";
  }

  /**
   * Gets the generation.
   * 
   * @return the generation
   */
  public String getGeneration() {
    return String.valueOf(evaluator.getAlgorithm().getGeneration());
  }

  /**
   * Callback method. Called from the gui.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void rotateLeft() {
    evaluator.getDragListener().rotateCamera(FastMath.DEG_TO_RAD * -15f);
  }

  /**
   * Callback method. Called from the gui.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void rotateRight() {
    evaluator.getDragListener().rotateCamera(FastMath.DEG_TO_RAD * 15f);
  }

  /**
   * Callback method. Called from the gui.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void rotateUp() {
    evaluator.getDragListener().vRotateCamera(FastMath.DEG_TO_RAD * -5f);
  }

  /**
   * Callback method. Called from the gui.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void rotateDown() {
    evaluator.getDragListener().vRotateCamera(FastMath.DEG_TO_RAD * 5f);
  }

}
